﻿//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.Core.js
//
/// <reference path="~/Scripts/DevTime.js" />
/// <reference path=~/Scripts/jquery-1.6.2-vsdoc.js" />
/// <reference path="~/Scripts/MicrosoftAjax-4.0.0.0.debug.js"/>
/// <reference path="~/Scripts/TFS/TFS.debug.js" />
/// <reference path="~/Scripts/TFS/TFS.diag.debug.js" />

TFS.module("TFS.Core", ["TFS.Diag", "TFS.Resources.Common"], function () {
    var antiForgeryToken,
        antiForgeryToken2,
		CommonResources = TFS.Resources.Common,
        Diag = TFS.Diag,
		OperationCanceledException = "TFS.Core.OperationCanceledException";

    Error.canceled = function () {
        var err = Error.create("Operation Canceled.", { name: OperationCanceledException });
        err.popStackFrame();
        return err;
    };

    function delegate(instance, method, data) {
        return function () {
            if (typeof (data) === "undefined") {
                return method.apply(instance, arguments);
            }
            else {
                var args = Array.prototype.slice.apply(arguments, [0]);

                if (data instanceof Array) {
                    args = args.concat(data);
                }
                else {
                    args.push(data);
                }

                return method.apply(instance, args);
            }
        };
    }

    function transformError(errorCallback, messageOrTransform, errorInfo) {
        /// <summary>Returns an error callback function, that wraps an input error callback, and invokes it
        /// after wrapping the input error passed to that function with a new error.</summary>
        /// <param name="errorCallback" type="function">The callback function to be invoked with the new wrapped error instance.</param>
        /// <param name="messageOrTransform" type="object">A string for  message for the new error being created, or a function that receives an error object and returns a new error.</param>
        /// <param name="errorInfo" type="Object">An optional object that contains extended information about the error.</param>
        /// <returns>A wrapper callback function that will transform the input error, or null if the input callback was not a function.</returns>
        var result = null,
			newError;

        if (typeof (messageOrTransform) !== "string" && !$.isFunction(messageOrTransform)) {
            throw new Error.argument("messageOrTransform", messageOrTransform);
        }

        // For API usability, if the callback is null, then return null (no-op)
        if ($.isFunction(errorCallback)) {
            result = function (error) {
                if (typeof (messageOrTransform) === "string") {
                    newError = Error.create(messageOrTransform, errorInfo);
                    newError.innerException = error;
                }
                else {
                    newError = messageOrTransform(error);
                }

                errorCallback(newError);
            };
        }

        return result;
    }

    function delay(instance, ms, method, data) {
        /// <summary>Executes the provided function after the specified amount of time</summary>
        /// <param name="instance" type="Object">Context to use when calling the provided function</param>
        /// <param name="ms" type="Integer">Delay in milliseconds to wait before executing the Function</param>
        /// <param name="method" type="Function">Method to execute</param>
        /// <param name="data" type="Array">(Optional) arguments to pass to the method</param>
        /// <returns type="DelayedFunction">The delayed function that was started</returns>

        var delayedFunc = new DelayedFunction(instance, ms, null, method, data);
        delayedFunc.start();
        return delayedFunc;
    }

    function throttledDelegate(instance, ms, method, data) {
        /// <summary>Creates a delegate that is delayed for the specified interval when invoked. Subsequent calls to the returned delegate reset the timer.</summary>
        /// <param name="instance" type="Object">Context to use when calling the provided function</param>
        /// <param name="ms" type="Integer">Delay in milliseconds to wait before executing the Function</param>
        /// <param name="method" type="Function">Method to execute</param>
        /// <param name="data" type="Array">(Optional) arguments to pass to the method</param>
        var delayedFunc = new DelayedFunction(instance, ms, "throttledDelegate", method, data);
        return delegate(delayedFunc, delayedFunc.reset);
    }

    function DelayedFunction(instance, ms, name, method, data) {
        /// <summary>Creates an object that can be used to delay-execute the specified method.</summary>
        /// <param name="instance" type="Object">Context to use when calling the provided function</param>
        /// <param name="ms" type="Integer">Delay in milliseconds to wait before executing the Function</param>
        /// <param name="name" type="String">Name to use when tracing this delayed function</param>
        /// <param name="method" type="Function">Method to execute</param>
        /// <param name="data" type="Array">(Optional) arguments to pass to the method</param>
        this._interval = ms;
        this._name = name;
        this._func = delegate(instance, method, data);
    }

    DelayedFunction.prototype = {
        _interval: 0,
        _func: null,
        _timeoutHandle: null,
        _name: null,

        start: function () {
            /// <summary>Starts the timer (if not already started) which will invoke the method once expired.</summary>
            var that = this;
            if (!this._timeoutHandle) {
                TFS.Diag.logTracePoint("Core.DelayedFunction.pending", this._name);
                this._timeoutHandle = window.setTimeout(function () {
                    delete that._timeoutHandle;
                    try {
                        that._func.call(that);
                    }
                    finally {
                        TFS.Diag.logTracePoint("Core.DelayedFunction.complete", [that._name, "complete"]);
                    }
                }, this._interval);
            }
        },

        reset: function () {
            /// <summary>Resets the timer (cancel, then re-start) which will invoke the method once expired.</summary>
            this.cancel();
            this.start();
        },

        cancel: function () {
            /// <summary>Cancels any pending operation (stops the timer).</summary>
            if (this._timeoutHandle) {
                window.clearTimeout(this._timeoutHandle);
                delete this._timeoutHandle;
                TFS.Diag.logTracePoint("Core.DelayedFunction.complete", [this._name, "canceled"]);
            }
        },

        invokeNow: function () {
            /// <summary>Invokes the method immediately (canceling an existing timer).</summary>
            this.cancel();
            this._func();
        },

        setDelay: function (ms) {
            /// <summary>Modifies the length of the delay timer (for subsequent starts).</summary>
            /// <param name="ms" type="Integer">Delay in milliseconds to wait before executing the Function</param>
            this._interval = ms;
        },

        setMethod: function (instance, method, data) {
            /// <summary>Modify the method being executed.</summary>
            /// <param name="instance" type="Object">Context to use when calling the provided function</param>
            /// <param name="method" type="Function">Method to execute</param>
            /// <param name="data" type="Array">(Optional) arguments to pass to the method</param>
            this._func = delegate(instance, method, data);
        },

        isPending: function () {
            /// <summary>Is the timer currently running (operation in progress)</summary>
            /// <returns type="Boolean">True if this operation is already in progress</returns>
            return this._timeoutHandle ? true : false;
        }
    };

    function Cancelable(context) {
        this.context = context;

        this._callbacks = [];
    }

    Cancelable.prototype = {
        canceled: false,
        context: null,
        _callbacks: null,
        perform: function (action) {
            if (!this.canceled && $.isFunction(action)) {
                action.call(this.context);
            }
        },
        wrap: function (action) {
            var that = this;
            return function () {
                if (!that.canceled && $.isFunction(action)) {
                    return action.apply(this, Array.prototype.slice.call(arguments, 0));
                }
            }
        },
        cancel: function () {
            var that = this;
            this.canceled = true;

            $.each(this._callbacks, function (i, callback) {
                callback.call(that.context);
            });
        },
        register: function (callback) {
            if ($.isFunction(callback)) {
                this._callbacks.push(callback);

                if (this.canceled) {
                    callback.call(this.context);
                }
            }
        }
    };

    function EventListBehavior() {
    }

    EventListBehavior.prototype = {
        _events: null,
        _fireEvent: function (eventName, sender, args) {
            /// <summary>Invoke the specified event passing the specified arguments.</summary>
            /// <param name="eventName">The event to invoke.</param>
            /// <param name="sender">The sender of the event.</param>
            /// <param name="args">The arguments to pass through to the specified event.</param>
            var i, evt;

            if (this._events) {
                // First getting the events attached to this command
                evt = this._events._getEvent(eventName);

                // Checking to see any handler exists for this command
                if (!evt || (evt.length === 0)) {
                    return;
                }

                evt = Array.clone(evt);

                // Executing each handler in reverse order
                for (i = evt.length - 1; i >= 0; i--) {
                    // If any handler returns false
                    if (evt[i](sender || this, args) === false) {
                        // Stop executing remaining handlers
                        return false;
                    }
                }
            }
        },
        attachEvent: function (eventName, handler) {
            if (!this._events) {
                this._events = new Sys.EventHandlerList();
            }

            this._events.addHandler(eventName, handler);
        },
        detachEvent: function (eventName, handler) {
            if (this._events) {
                this._events.removeHandler(eventName, handler);
            }
        }
    };

    function TransitionTime(timeOfDay, month, week, day, dayOfWeek, isFixedDateRule) {
        if (month < 1 || month > 12) {
            throw new Error.argumentOutOfRange("month", month);
        }

        if (day < 1 || day > 31) {
            throw new Error.argumentOutOfRange("day", day);
        }

        if (week < 1 || week > 5) {
            throw new Error.argumentOutOfRange("week", week);
        }

        if (dayOfWeek < 0 || dayOfWeek > 6) {
            throw new Error.argumentOutOfRange("dayOfWeek", dayOfWeek);
        }

        if (timeOfDay instanceof Date) {
            if (timeOfDay.getFullYear() !== 1 || timeOfDay.getMonth() !== 1 || timeOfDay.getDay() !== 1 || timeOfDay.getTime() % 0x2710 !== 0) {
                throw new Error.argument("timeOfDay", timeOfDay);
            }
        }
        else {
            throw new Error.argumentType(dayOfWeek, Object.getType(dayOfWeek), Date);
        }

        this.day = day;
        this.month = month;
        this.week = week;
        this.dayOfWeek = dayOfWeek;
        this.timeOfDay = timeOfDay;
        this.isFixedDateRule = isFixedDateRule || false;
    }

    TransitionTime.prototype = {
        day: 0,
        dayOfWeek: 0,
        isFixedDateRule: true,
        month: 0,
        timeOfDay: null,
        week: 0
    };

    function AdjustmentRule() {
    }

    Date.prototype.ago = function () {
        var i, len, result,
                minute = 60,
                hour = minute * 60,
                day = hour * 24,
                week = day * 7,
                month = (day * 365) / 12,
                year = day * 365,
                date = new Date(),
                date,
                diff,
                step,
                steps = [{ limit: minute, format: CommonResources.AgoLessThanAMinute },
                    { limit: minute * 1.5, format: CommonResources.AgoAMinute },
                    { limit: hour, format: CommonResources.AgoMinutes, arg: minute },
                    { limit: hour * 1.5, format: CommonResources.AgoAnHour },
                    { limit: day, format: CommonResources.AgoHours, arg: hour },
                    { limit: day * 1.5, format: CommonResources.AgoADay },
                    { limit: week, format: CommonResources.AgoDays, arg: day },
                    { limit: week * 1.5, format: CommonResources.AgoAWeek },
                    { limit: month, format: CommonResources.AgoWeeks, arg: week },
                    { limit: month * 1.5, format: CommonResources.AgoAMonth },
                    { limit: year, format: CommonResources.AgoMonths, arg: month },
                    { limit: year * 1.5, format: CommonResources.AgoAYear },
                    { limit: Number.POSITIVE_INFINITY, format: CommonResources.AgoYears, arg: year }];

        if (arguments.length > 0 && arguments[0] instanceof Date) {
            // This might be used for testing purposes
            date = arguments[0];
        }

        // Getting the difference between now and the specified date
        diff = date - this;
        // Converting diff to seconds
        diff /= 1000;

        for (i = 0, len = steps.length; i < len; i++) {
            step = steps[i];
            if (diff < step.limit) {
                if (step.arg) {
                    result = String.format(step.format, Math.round(diff / step.arg));
                }
                else {
                    result = step.format;
                }
                break;
            }
        }

        if (!result) {
            result = this.toString();
        }

        return result;
    };

    Date.prototype.friendly = function () {
        var i, len, result,
            day = 60 * 60 * 24,
            now = new Date(),
            diff,
            step,
            steps,
            firstDayOfWeek;

        if (arguments.length > 0 && arguments[0] instanceof Date) {
            // This might be used for testing purposes
            now = arguments[0];
        }

        firstDayOfWeek = new Date(now.getFullYear(), now.getMonth(),
            now.getDate() - now.getDay() + Sys.CultureInfo.CurrentCulture.dateTimeFormat.FirstDayOfWeek);

        steps = [
            {
                limit: day,
                format: function (date) {
                    return date.ago(now);
                }
            },
            {
                limit: (now - firstDayOfWeek) / 1000,
                format: function (date) {
                    return date.localeFormat('dddd');
                }
            },
            {
                limit: Number.POSITIVE_INFINITY,
                format: function (date) {
                    return date.localeFormat('d');
                }
            }];

        // Getting the difference between now and the specified date in seconds
        diff = (now - this) / 1000;

        for (i = 0, len = steps.length; i < len; i++) {
            step = steps[i];
            if (diff < step.limit && step.limit > 0) {
                result = step.format(this);
                break;
            }
        }

        if (!result) {
            result = this.toString();
        }

        return result;
    };

    function getAntiForgeryTokenValue() {
        if (!antiForgeryToken) {
            antiForgeryToken = $("input[name=__RequestVerificationToken]").val();
        }
        return antiForgeryToken;
    }

    function getAntiForgeryTokenValue2() {
        if (!antiForgeryToken2) {
            antiForgeryToken2 = $("input[name=__RequestVerificationToken2]").val();
        }
        return antiForgeryToken2;
    }

    function getAntiForgeryToken() {
        var $token, tokenValue2;
        
        $token = $("<input />").attr("type", "hidden").attr("name", "__RequestVerificationToken").val(getAntiForgeryTokenValue());
        tokenValue2 = getAntiForgeryTokenValue2();
        if (tokenValue2) {
            $token.after($("<input />").attr("type", "hidden").attr("name", "__RequestVerificationToken2").val(getAntiForgeryTokenValue2()));
        }
        return $token;
    }

    function setAntiForgeryToken(form) {
        var $form = $(form);

        if ($form.find("input[name=__RequestVerificationToken]").length === 0) {
            return $form.append(getAntiForgeryToken());
        }
    }

    function StringBuilder() {
        this._textBuilder = [];
    }

    StringBuilder.prototype = {
        _textBuilder: null,

        append: function (text) {
            /// <summary>Appends the specified text to the end of the string buffer.</summary>
            /// <param name="text" type="String">The text to append.</param>

            if (typeof text === "string") {
                this._textBuilder[this._textBuilder.length] = text;
            }
        },

        appendNewLine: function () {
            /// <summary>Appends a new-line to the current text buffer.</summary>
            this.append(StringConstants.newLine);
        },

        toString: function () {
            /// <summary>Concatenates all text in the string buffer into a single string value.</summary>
            return this._textBuilder.join("");
        }
    };

    StringConstants = function () {
    }

    StringConstants.extend({
        newLine: "\r\n",
        tab: "\t"
    });

    function OperationQueue() {
        /// <summary>Allows for sequential processing of asyncronous operations.</summary>
        this._operationQueue = [];
    }

    OperationQueue.prototype = {
        _operationQueue: null,
        _isProcessingOperation: false,

        queueOperation: function (operation) {
            /// <summary>Queues the provided operation.  Operations are processed sequentially.</summary>
            /// <param name="operation" type="function">
            /// Function for the operation to be performed.  The function should have the following signature:
            ///         function operation(completedCallback)
            ///
            /// The completed callback needs to be invoked when the operation is completed in order to allow subsequent
            /// operations to be performed.
            /// </param>

            Diag.assertParamIsFunction(operation, "operation");

            // Add the operation to the queue.
            this._operationQueue.push(operation);

            // Attempt to start the operation.
            this._processQueue();
        },

        _processQueue: function () {
            /// <summary>Begins processing the next operation in the queue if there is not one already in progress.</summary>

            var that = this;

            // If there is not a current operation being processed and there are operations
            // in the queue, process them.
            if (!this._isProcessingOperation && this._operationQueue.length > 0) {
                // Indicate that an operation is being processed.
                this._isProcessingOperation = true;

                // Invoke the next operation on a delay to avoid blocking the caller on the operation and avoid
                // the potential of having the operations stack grow large.
                delay(this, 0, function () {
                    var operation,
                        hasCompleted = false;

                    function completedCallback() {
                        /// <summary>Invoked when the operation has completed to kick off the next operation in the queue.</summary>

                        // If the complete callback has already been invoked, assert.
                        Diag.assert(!hasCompleted, "The operation has already invoked the completed callback.");

                        if (!hasCompleted) {
                            hasCompleted = true;

                            // Start the next operation in the queue.
                            that._isProcessingOperation = false;
                            that._processQueue();
                        }
                    }

                    // Remove the operation from the queue and execute it.
                    operation = that._operationQueue.shift();
                    operation(completedCallback);
                })
            }
        }
    };

    return {
        OperationCanceledException: OperationCanceledException,
        delegate: delegate,
        transformError: transformError,
        delay: delay,
        throttledDelegate: throttledDelegate,
        DelayedFunction: DelayedFunction,
        Cancelable: Cancelable,
        EventListBehavior: EventListBehavior,
        getAntiForgeryTokenValue: getAntiForgeryTokenValue,
        getAntiForgeryTokenValue2: getAntiForgeryTokenValue2,
        setAntiForgeryToken: setAntiForgeryToken,
        StringBuilder: StringBuilder,
        StringConstants: StringConstants,
        OperationQueue: OperationQueue
    };
});

// SIG // Begin signature block
// SIG // MIIamQYJKoZIhvcNAQcCoIIaijCCGoYCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFJWI8K8qGqKD
// SIG // Ji2nZ6nG/0BNfJtuoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSMMIIEiAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGuMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBQcKRsScMjSRfyu3gU8s72h
// SIG // cqaEXTBOBgorBgEEAYI3AgEMMUAwPqAkgCIAVABGAFMA
// SIG // LgBDAG8AcgBlAC4AZABlAGIAdQBnAC4AagBzoRaAFGh0
// SIG // dHA6Ly9taWNyb3NvZnQuY29tMA0GCSqGSIb3DQEBAQUA
// SIG // BIIBACb91FohxXNJ9C/v3iMPdgIgP3KFjxQYtm9TIhW2
// SIG // LNcWf2R/A6YjcIZdx0jKVOK7LPHNsX5/kl6+uxrNzFS7
// SIG // edWiHkLKfVJLhG71QVTmLYSnWF7fBkqa0RQ7/4oIQbbs
// SIG // rHfyUQNLQNWdMRRIDEMfQlJhOD6vYdaK/mYFgDqhGhjA
// SIG // g0f1BtzBiDgNnRlHWx6wPEhcOFN25KqkEVoki+aay9t9
// SIG // d3uqeRAJwPROPfJo8RqCJtlCAklcF0YA0b2W5Dzdizso
// SIG // WI4ZaX//6rFJchC2KqvptEjfRkXqYrJA1HkrJqPthD/G
// SIG // RhTBk6YcvU+wWEPHo2rgNFUzDeP+17gV+ZDnFB2hggIf
// SIG // MIICGwYJKoZIhvcNAQkGMYICDDCCAggCAQEwgYUwdzEL
// SIG // MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
// SIG // EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9z
// SIG // b2Z0IFRpbWUtU3RhbXAgUENBAgphApJKAAAAAAAgMAkG
// SIG // BSsOAwIaBQCgXTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcN
// SIG // AQcBMBwGCSqGSIb3DQEJBTEPFw0xMzAzMTUwNjMzNTVa
// SIG // MCMGCSqGSIb3DQEJBDEWBBRhQoDv4RRJTNdDtEjLvz4P
// SIG // wT7SojANBgkqhkiG9w0BAQUFAASCAQBVXQWx+4tyl15/
// SIG // PEZJowbkN67RBAagH6sS0HAaRNnmd0sUGu3fOv2Y95Kh
// SIG // 5D3ppyUhC186aQugWWSlhyIkW4C0A2wC0FhFpDB4ESoz
// SIG // OW3sBBkCEieS/q4P5z7KeV9dnJ8R3ld1YoSnG4UY8h6n
// SIG // oIMwnj2bijk3Nyn4XmFp2fsoCjLdn0LNPBWM6xtn+DSj
// SIG // JYKxTxtSjbpXkJERle8M1IvTVk3QJTAGOCLyRzl76h5t
// SIG // rg5W5FgNpSzkI6K9mmquijP5s6zdf+c+ADGbGZtvN4f5
// SIG // eAjWX8/DIgbP/PI45+KW5+DpYmYh5VsQWHs540lgEK1W
// SIG // icHnrMptrGqGI+3KdAYd
// SIG // End signature block
